package easik.states;

import javax.swing.BorderFactory;

import org.jgraph.graph.AttributeMap;
import org.jgraph.graph.DefaultGraphCell;
import org.jgraph.graph.GraphConstants;

import easik.Easik;
import easik.sketch.Sketch;
import easik.sketch.constraint.Constraint;
import easik.sketch.edge.GuideEdge;
import easik.sketch.edge.SketchEdge;
import easik.sketch.vertex.EntityNode;


/** 
 *	This is the base state. It contains stubs and abstract methods for allowing a 
 *	class to be treated like a state, and to take part in the State Manager
 *
 * @author Rob Fletcher 2005
 * @author Vera Ranieri 2006
 * @author Kevin Green 2006
 * @version 2006-08-02 Kevin Green
 */
public abstract class EasikState {
	/**
	 * Hook to record when selection has been updated
	 *
	 */
	public void selectionUpdated() {};		
	/**
	 * Hook to record when state has been pushed on
	 *
	 */
	public abstract void pushedOn();	
	/**
	 * Hook to record when state has been popped off
	 *
	 */
	public abstract void poppedOff();	
	/**
	 * @return The string representation of this state
	 */
	public abstract String toString();
	/**
	 * Records state of the next button
	 */
	private boolean _nextOn = false;
	/**
	 * Records the state of the cancel button
	 */
	private boolean _cancelOn = false;	
	/**
	 * Records the state of the finish button
	 */
	private boolean _finishOn = false;
	/**
	 * The current sketch
	 */
	protected Sketch _ourSketch;

	/**
	 * When this state gets the top spot again, this gets executed. Restores buttons
	 * and labels. If overridden, be sure to at call super.gotfocus() first.
	 */
	public void gotFocus() {
		setCancelButton( _cancelOn );
		setNextButton( _nextOn );
		setFinishButton(_finishOn);
		Easik.getInstance().getFrame().setStateString( this.toString() );		
	}
	
	/**
	 * Optional hook for when the cancel button is clicked
	 *
	 */
	public void cancelClicked() {
	}
	
	/**
	 * Optional hook for when the next button is clicked
	 *
	 */
	public void nextClicked() {
	}
	
	/**
	 * Optional hook for when the finish button is clicked.
	 * 
	 * @since 2006-05-30 Vera Ranieri
	 */
	public void finishClicked(){	
	}
	
	/**
	 * Use this function to toggle the usability of the cancel button. It
	 * remembers the state in a separate boolean so it can be restored.
	 * 	 
	 * @param inButton Status of the button
	 */
	protected void setCancelButton( boolean inButton ) {
		_cancelOn = inButton;
		Easik.getInstance().getFrame().getCancelButton().setEnabled( inButton );
		if(_cancelOn || _nextOn || _finishOn)
			Easik.getInstance().getFrame().setButtonPaneVisibility(true);
		else
			Easik.getInstance().getFrame().setButtonPaneVisibility(false);
	}
	
	/**
	 * Use this function to toggle the usability of the ok button. It
	 * remembers the state in a separate boolean so it can be restored.
	 * 	 
	 * @param inButton Status of the button
	 */
	protected void setNextButton( boolean inButton ) {
		_nextOn = inButton;
		Easik.getInstance().getFrame().getNextButton().setEnabled( inButton );
		if(_cancelOn || _nextOn || _finishOn)
			Easik.getInstance().getFrame().setButtonPaneVisibility(true);
		else
			Easik.getInstance().getFrame().setButtonPaneVisibility(false);
	}	
	
	/**
	 * Use this function to toggle the usability of the finish button. It
	 * remembers the state in a separate boolean so it can be restored.
	 * 
	 * @param inButton Status of the button
	 * @since 2006-05-30 Vera Ranieri
	 */
	protected void setFinishButton(boolean inButton){
		_finishOn = inButton;
		Easik.getInstance().getFrame().getFinishButton().setEnabled( inButton );
		if(_cancelOn || _nextOn || _finishOn)
			Easik.getInstance().getFrame().setButtonPaneVisibility(true);
		else
			Easik.getInstance().getFrame().setButtonPaneVisibility(false);
	}
	
	/**
	 * Resets the graph colors to what they are by default.
	 */
	public void resetColors(){
		Object entireSet[] = Easik.getInstance().getFrame().getSketch().getRoots();
		for(int i=0; i<entireSet.length; i++){
			DefaultGraphCell curObject = (DefaultGraphCell) entireSet[i];
			AttributeMap myMap = ((DefaultGraphCell)entireSet[i]).getAttributes();
			if(curObject.getUserObject() instanceof EntityNode){
				GraphConstants.setBorder(myMap, BorderFactory.createLineBorder(Easik.getInstance().getIni().getENTITY_BORDER_COLOR(), 1));
				GraphConstants.setForeground(myMap, Easik.getInstance().getIni().getENTITY_FG_COLOR());
				GraphConstants.setBackground(myMap, Easik.getInstance().getIni().getENTITY_BG_COLOR());
			}
			else if(curObject.getUserObject() instanceof Constraint){
				GraphConstants.setBorder(myMap, BorderFactory.createLineBorder(Easik.getInstance().getIni().getCONSTRAINT_BORDER_COLOR(), 1));
				GraphConstants.setForeground(myMap, Easik.getInstance().getIni().getCONSTRAINT_FG_COLOR());
				GraphConstants.setBackground(myMap, Easik.getInstance().getIni().getCONSTRAINT_BG_COLOR());
			}
			else if(curObject.getUserObject() instanceof SketchEdge){
				GraphConstants.setLineWidth(myMap, 1);
				GraphConstants.setForeground(myMap, Easik.getInstance().getIni().getENTITY_FG_COLOR());
				if(!((SketchEdge)curObject.getUserObject()).getInjective()){
					GraphConstants.setLineColor(myMap, Easik.getInstance().getIni().getSIMPLE_EDGE_COLOR());
				}
				else{
					GraphConstants.setLineColor(myMap, Easik.getInstance().getIni().getFLETCHED_EDGE_COLOR());
				}
			}
			else if(curObject.getUserObject() instanceof GuideEdge){
				if(((GuideEdge)curObject.getUserObject()).IsHighlighted()){
					GraphConstants.setLineColor(myMap, Easik.getInstance().getIni().getVIRTUAL_HIGHLIGHTED_EDGE_COLOR());
				}
				else{
					GraphConstants.setLineColor(myMap, Easik.getInstance().getIni().getVIRTUAL_EDGE_COLOR());
				}
			}
			((DefaultGraphCell)entireSet[i]).setAttributes(myMap);
		}
	}
}
